program STEFFENSEN;
{--------------------------------------------------------------------}
{  Alg2'78.pas   Pascal program for implementing Algorithm 2.7-8     }
{                                                                    }
{  NUMERICAL METHODS: Pascal Programs, (c) John H. Mathews 1995      }
{  To accompany the text:                                            }
{  NUMERICAL METHODS for Math., Science & Engineering, 2nd Ed, 1992  }
{  Prentice Hall, Englewood Cliffs, New Jersey, 07632, U.S.A.        }
{  Prentice Hall, Inc.; USA, Canada, Mexico ISBN 0-13-624990-6       }
{  Prentice Hall, International Editions:   ISBN 0-13-625047-5       }
{  This free software is compliments of the author.                  }
{  E-mail address:       in%"mathews@fullerton.edu"                  }
{                                                                    }
{  Algorithm 2.7 (Steffensen's Acceleration).                        }
{  Section   2.5, Aitken's Process & Steffensen's & Muller's Methods,}
{  Page 96                                                           }
{  Algorithm 2.8 (Muller's Method).                                  }
{  Section   2.5, Aitken's Process & Steffensen's & Muller's Methods,}
{  Page 97                                                           }
{--------------------------------------------------------------------}

  uses
    crt;

  const
    Max = 99;
    Vmax = 102;
    FunMax = 9;

  type
    PVECTOR = array[0..Vmax] of real;
    LETTERS = string[200];
    STATUS = (Computing, Done, Working);

  var
    Cond, FunType, Inum, K, Meth, Sub: integer;
    Dp, Delta, Epsilon, P0, P1, P2, P3, Rnum, Z: real;
    VP: PVECTOR;
    Satisfied: BOOLEAN;
    Mess: LETTERS;
    Stat, State: STATUS;
    Ans: CHAR;

  function F (X: real): real;
  begin
    case FunType of
      1: 
        F := X * X * X - 3 * X + 2;
      2: 
        F := X * X * X - 3 * X - 2;
      3: 
        F := X * X * X - X - 2;
      4: 
        F := (X - 1) * LN(X);
      5: 
        F := COS(X) - 1;
      6: 
        F := X - SIN(X);
      7: 
        F := 4 * X * X - EXP(X);
      8: 
        F := 1 + 2 * X - SIN(X) / COS(X);
      9: 
        F := 3 * COS(X) + 2 * SIN(X);
    end;
  end;

  function F1 (X: real): real;
  begin
    case FunType of
      1: 
        F1 := 3 * X * X - 3;
      2: 
        F1 := 3 * X * X - 3;
      3: 
        F1 := 3 * X * X - 1;
      4: 
        F1 := 1 - 1 / X + LN(X);
      5: 
        F1 := -SIN(X);
      6: 
        F1 := 1 - COS(X);
      7: 
        F1 := 8 * X - EXP(X);
      8: 
        F1 := 2 - SIN(X) / COS(X) / COS(X);
      9: 
        F1 := -3 * SIN(X) + 2 * COS(X);
    end;
  end;

  procedure PRINTFUNCTION (FunType: integer);
  begin
    case FunType of
      1: 
        WRITELN('F(X) = X*X*X - 3*X + 2');
      2: 
        WRITELN('F(X) = X*X*X - 3*X - 2');
      3: 
        WRITELN('F(X) = X*X*X - X - 2');
      4: 
        WRITELN('F(X) = (X - 1)*LN(X)');
      5: 
        WRITELN('F(X) = COS(X) - 1');
      6: 
        WRITELN('F(X) = X - SIN(X)');
      7: 
        WRITELN('F(X) = 4*X*X - EXP(X)');
      8: 
        WRITELN('F(X) = 1 + 2*X - TAN(X)');
      9: 
        WRITELN('F(X) = 3*COS(X) + 2*SIN(X)');
    end;
  end;

  procedure STEFFEN (P0, Delta, Epsilon, Max: real; var P3, Dp: real; var Cond, K: integer);
    label
      999;
    const
      Small = 1E-20;
    var
      D1, D2, DF0, DF1, P1, P2, RelErr, Y3: real;
  begin
    K := 0;
    Cond := 0;
    P3 := P0;
    P2 := P0 + 1;
    P1 := P0 + 2;
    VP[0] := P3;    {Store the approximations}
    while (K < Max) and (Cond = 0) do
      begin
        P0 := P3;
        DF0 := F1(P0);
        if DF0 <> 0 then                           {Check division by  0}
          P1 := P0 - F(P0) / DF0                       {First Newton iterate}
        else
          begin
            Cond := 1;
            Dp := P3 - P2;
            P3 := P0;
            goto 999;
          end;
        DF1 := F1(P1);
        if DF1 <> 0 then                          {Check  division by  0}
          P2 := P1 - F(P1) / DF1                      {Second Newton iterate}
        else
          begin
            Cond := 1;
            Dp := P1 - P0;
            P3 := P1;
            goto 999;
          end;
        D1 := (P1 - P0) * (P1 - P0);                                  {Form the }
        D2 := P2 - 2 * P1 + P0;                                     {Differences}
        if D2 = 0 then
          begin                                    {Check division by  0}
            Cond := 1;
            Dp := P2 - P1;
            P3 := P2;
          end
        else
          begin
            P3 := P0 - D1 / D2;                          {Aitken's improvement}
            Dp := P3 - P2;
          end;
        Y3 := F(P3);
        RelErr := ABS(Dp);  { /(ABS(P3)+Small); }
   {Convergence criterion}
        if (RelErr <= Delta) then
          Cond := 2;
        if (ABS(Y3) < Epsilon) then
          Cond := 3;
        if (RelErr <= Delta) and (ABS(Y3) < Epsilon) then
          Cond := 4;
999:
        K := K + 1;                                   {Increment the counter}
        VP[K] := P3;  {Store the approximations}
      end;
  end;

  procedure MULLER (P0, P1, P2: real; Delta, Epsilon, Max: real; var P3, Z: real; var Satisfied: BOOLEAN);
    const
      Small = 1E-20;
    var
      A, B, C, Det, Disc, H0, H1, E0, E1: real;
      Y0, Y1, Y2, RelErr, U, V: real;
  begin
    K := 0;
    Satisfied := False;
    Y0 := F(P0);                                {Compute function values}
    Y1 := F(P1);
    Y2 := F(P2);
    VP[0] := P0;  {Store the approximations}
    VP[1] := P1;
    VP[2] := P2;
    while (K < Max) and (Satisfied = False) do
      begin
        H0 := P0 - P2;                                     {Form differences}
        H1 := P1 - P2;
        C := Y2;
        E0 := Y0 - C;
        E1 := Y1 - C;
        Det := H0 * H1 * (H0 - H1);                         {Compute determinants}
        A := (E0 * H1 - H0 * E1) / Det;                            {and solve the}
        B := (H0 * H0 * E1 - H1 * H1 * E0) / Det;                      {linear system}
        if B * B > 4 * A * C then
          Disc := SQRT(B * B - 4 * A * C)        {This suppresses complex roots}
        else
          DISC := 0;
        if B < 0 then                                         {Find the}
          Disc := -Disc;                                   {smallest root}
        if (B + Disc) <> 0 then                         {of the quadratic}
          Z := -2 * C / (B + Disc)
        else
          Z := 0;
        P3 := P2 + Z;
        if ABS(P3 - P1) < ABS(P3 - P0) then                         {Sort to}
          begin                                                {make the}
            U := P1;                                        {values P0,P1}
            P1 := P0;                                       {closest to P3}
            P0 := U;
            V := Y1;
            Y1 := Y0;
            Y0 := V;
          end;
        if ABS(P3 - P2) < ABS(P3 - P1) then
          begin
            U := P2;
            P2 := P1;
            P1 := U;
            V := Y2;
            Y2 := Y1;
            Y1 := V;
          end;
        P2 := P3;                                          {Update iterate}
        Y2 := F(P2);
        RelErr := ABS(Z) / (ABS(P2) + Small);                  {Relative error}
        if (RelErr <= Delta) and (ABS(Y2) < Epsilon) then     {Check for}
          Satisfied := True;                                  {convergence}
        if ABS(P2 - P1) = 0 then
          Satisfied := True;
        if ABS(P2 - P0) = 0 then
          Satisfied := True;
        K := K + 1;                                       {Increment counter}
        VP[K + 2] := P2;  {Store the approximations}
      end;
    K := K + 1;
  end;

  procedure MESSAGE (var Meth: integer; var Delta, Epsilon: real);
    var
      I: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITELN('                      STEFFENSEN  AND  MULLER  METHODS');
    WRITELN;
    WRITELN;
    WRITELN('         An acceleration method will be used to find the roots');
    WRITELN;
    WRITELN;
    WRITELN('     of the equation  F(x) = 0 .');
    WRITELN;
    WRITELN;
    WRITELN('     You have a choice of two methods:');
    WRITELN;
    WRITELN;
    WRITELN('                < 1 > Steffensen`s method');
    WRITELN;
    WRITELN;
    WRITELN('                < 2 > Muller`s method');
    WRITELN;
    WRITELN;
    WRITELN;
    Mess := '                SELECT your method  < 1 or 2 > ?  ';
    WRITE(Mess);
    READLN(Meth);
    if Meth < 1 then
      Meth := 1;
    if Meth > 2 then
      Meth := 2;
    CLRSCR;
    WRITELN;
    case Meth of
      1: 
        begin
          WRITELN('An initial starting value p  must be given. Then the Newton-Raphson function');
          WRITELN('                           0');
          WRITELN;
          WRITELN('G(x) = x - F(x)/F`(x)  is used to compute the next two iterates  p   and  p :');
          WRITELN('                                                                  1        2 ');
          WRITELN('       p  = G(p )  and  p  = G(p ).');
          WRITELN('        1      0         2      1');
          WRITELN;
          WRITELN('Then Aitken`s formula is used to compute the next iterate  p ');
          WRITELN('                                                            3');
          WRITELN('                             2                  ');
          WRITELN('       p   =  p   -  (p - p ) /(p  - 2p  + p ) .');
          WRITELN('        3      0       1   0     2     1    0   ');
          WRITELN;
          WRITELN('Iteration is continued by alternating with two steps of the Newton-Raphson');
          WRITELN;
          WRITELN('formula and one step of Aitken`s formula.');
        end;
      2: 
        begin
          WRITELN('     Three starting values  p  , p  and p   must be given.');
          WRITELN('                             0    1      2');
          WRITELN;
          WRITELN('Then a parabola is fit through the three points  (p ,F(p ))  k=0,1,2.');
          WRITELN('                                                   k    k ');
          WRITELN;
          WRITELN('The smallest root of the resulting quadratic is used for  p  .');
          WRITELN('                                                           3');
          WRITELN;
          WRITELN('Iteration is continued by using the latest three points');
          WRITELN;
          WRITELN('generate the next parabola.');
        end;
    end;
    WRITELN;
    WRITELN;
    WRITE('Press the <ENTER> key.  ');
    READLN(Ans);
    CLRSCR;
    WRITELN;
    WRITELN;
    case Meth of
      1: 
        begin
          WRITELN('     Convergence is declared when the difference in iterates in the');
          WRITELN;
          WRITELN('Aitken calculation is small, or the function value is small, i.e.');
          WRITELN;
          WRITELN;
          WRITELN('     |p  - p   | < Delta    OR    |F(p )| < Epsilon.');
          WRITELN('       N    N-3                       N       ');
        end;
      2: 
        begin
          WRITELN('     Convergence is declared when the difference in consecutive iterates');
          WRITELN;
          WRITELN('is small, or the function value is small, i.e.');
          WRITELN;
          WRITELN;
          WRITELN('     |p  - p   | < Delta    OR    |F(p )| < Epsilon.');
          WRITELN('       N    N-1                       N       ');
        end;
    end;
    WRITELN;
    WRITELN;
    case Meth of
      1: 
        begin
          WRITELN('     Now give values for  Delta , Epsilon and p .');
          WRITELN('                                               0');
        end;
      2: 
        begin
          WRITELN('     Now give values for  Delta , Epsilon and p  , p  and p .');
          WRITELN('                                               0    1      2');
        end;
    end;
    WRITELN;
    Mess := '     ENTER  the  tolerance   Delta = ';
    Delta := 0.000000001;
    WRITE(Mess);
    READLN(Delta);
    Delta := ABS(Delta);
    if (Delta < 0.000000001) then
      Delta := 0.000000001;
    WRITELN;
    WRITELN;
    Mess := '     ENTER the tolerance   Epsilon = ';
    Epsilon := 0.000000001;
    WRITE(Mess);
    READLN(Epsilon);
    Epsilon := ABS(Epsilon);
    if (Epsilon < 0.000000001) then
      Epsilon := 0.000000001;
  end;

  procedure GETFUN (var FunType: integer);
    var
      I, K: integer;
  begin
    CLRSCR;
    case Meth of
      1:
        WRITELN('     You chose Steffensen`s method to find roots of  F(X) = 0.');
      2: 
        WRITELN('     You chose Muller`s method to find roots of  F(X) = 0.');
    end;
    WRITELN;
    WRITELN;
    for K := 1 to FunMax do
      begin
        WRITE('     <', K : 2, ' >   ');
        PRINTFUNCTION(K);
        WRITELN;
      end;
    Mess := '             SELECT your function  < 1 - 9 > ?  ';
    WRITE(Mess);
    READLN(FunType);
    if FunType < 1 then
      FunType := 1;
    if FunType > FunMax then
      FunType := FunMax;
  end;

  procedure GETPOINTS (var P0, P1, P2: real; Meth: integer);
    var
      T: real;
  begin
    CLRSCR;
    WRITELN;
    WRITELN;
    WRITE('     You chose ');
    if Meth = 1 then
      WRITE('Steffensen`s')
    else
      WRITE('Muller`s');
    WRITELN(' method to find a zero of the function:');
    WRITELN;
    WRITE('     ');
    PRINTFUNCTION(FunType);
    WRITELN;
    case Meth of
      1: 
        begin
          WRITELN('     One  starting  value  p   is  required.');
          WRITELN('                            0');
        end;
      2: 
        begin
          WRITELN('     Three starting values p , p  and p .');
          WRITELN('                            0   1      2');
        end;
    end;
    WRITELN;
    WRITELN;
    Mess := '          ENTER the value  p0 = ';
    WRITE(Mess);
    READLN(P0);
    WRITELN;
    P1 := P0 + 1;
    P2 := P0 + 2;
    if Meth = 2 then
      begin
        WRITELN;
        WRITELN;
        Mess := '          ENTER the value  p1 = ';
        WRITE(Mess);
        READLN(P1);
        WRITELN;
        if P1 = P0 then
          begin
            if P0 = 0 then
              P1 := 0.001
            else
              P1 := P0 * 1.001 + 0.001 * P0 / ABS(P0);
          end;
        WRITELN;
        WRITELN;
        Mess := '          ENTER the value  p2 = ';
        WRITE(Mess);
        READLN(P2);
        WRITELN;
        if (P2 = P0) or (P2 = P0) then
          begin
            if P0 = 0 then
              P2 := 0.001
            else
              P2 := P0 * 1.001 + 0.001 * P0 / ABS(P0);
            if P2 = P1 then
              begin
                if P1 = 0 then
                  P2 := 0.001
                else
                  P2 := P1 * 1.001 + 0.001 * P1 / ABS(P1);
              end;
          end;
      end;
    WRITELN;
  end;                                      {End of PROCEDURE GETPOINT}

  procedure RESULTS (P0, P3, Dp: real; Cond, K: integer);
  begin
    CLRSCR;
    WRITELN;
    WRITELN('     Steffensen`s acceleration of the Newton-Raphson method');
    WRITELN;
    WRITELN('was used to find a zero of the function');
    WRITELN;
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN('Starting with the approximation  p  =', P0 : 15 : 7);
    WRITELN('                                  0   ');
    WRITELN;
    WRITELN('After ', K : 1, ' iterations an approximate value of the zero is');
    WRITELN;
    WRITELN('  P  =', P3 : 15 : 7);
    WRITELN;
    WRITELN(' DP  =', ABS(Dp) : 15 : 7, '    is an estimate of the accuracy.');
    WRITELN;
    WRITELN('    F(', P3 : 15 : 7, '  ) =', F(P3) : 15 : 7);
    WRITELN;
    if F(P3) = 0 then
      begin
        WRITELN('The computed function value is exactly zero!');
        WRITELN;
      end;
    case Cond of
      0: 
        begin
          WRITELN('Convergence is doubtful because');
          WRITELN;
          WRITELN('the maximum number of iterations was exceeded.');
        end;
      1: 
        begin
          WRITELN('Convergence is doubtful because division by zero was encountered.');
        end;
      2: 
        begin
          WRITELN('The approximation P is within the desired tolerance.');
        end;
      3: 
        begin
          WRITELN('The function value F(P) is within the desired tolerance.');
        end;
      4: 
        begin
          WRITELN('The approximation P and the function value ');
          WRITELN('F(P) are both within the desired tolerances.');
        end;
    end;
  end;

  procedure RESULTM (P0, P1, P2, P3, Z: real; K: integer; var Satisfied: BOOLEAN);
    var
      Y3: real;
  begin
    Y3 := F(P3);
    CLRSCR;
    WRITELN;
    WRITELN('Muller`s method was used to find a zero of the function');
    WRITELN;
    PRINTFUNCTION(FunType);
    WRITELN;
    WRITELN('The three initial approximations were:');
    WRITELN;
    WRITELN('p  = ', P0 : 15 : 7, '    p  = ', P1 : 15 : 7, '    p  = ', P2 : 15 : 7);
    WRITELN(' 0                       1                       2   ');
    WRITELN;
    WRITELN('After ', K : 1, ' iterations an approximate value for a zero is:');
    WRITELN;
    WRITELN(' P =', P3 : 15 : 7);
    WRITELN;
    WRITELN('  F(', P3 : 15 : 7, '  ) = ', Y3 : 15 : 7);
    WRITELN;
    WRITELN('    ', ABS(Z) : 15 : 7, '   is the estimated accuracy');
    WRITELN;
    if Y3 = 0 then
      begin
        WRITELN('The computed function value is exactly zero!');
        WRITELN;
      end;
    if Satisfied = True then
      begin
        WRITELN('The zero was found and is within the desired tolerance.');
      end
    else
      begin
        WRITELN('Convergence is doubtful because the');
        WRITELN('maximum number of iterations was exceeded.');
      end;
  end;

  procedure PRINTAPPROXS;
    var
      J: integer;
  begin
    CLRSCR;
    WRITELN;
    WRITELN('           k             p                   F(p ) ');
    WRITELN('                          k                     k  ');
    WRITELN('         -------------------------------------------------');
    WRITELN;
    for J := 0 to K do
      begin
        WRITELN('          ', J : 2, '     ', VP[J] : 15 : 7, '     ', F(VP[J]) : 15 : 7);
        WRITELN;
        if J mod 11 = 8 then
          begin
            WRITE('                  Press the <ENTER> key. ');
            READLN(Ans);
            WRITELN;
          end;
      end;
    WRITELN;
    WRITE('                  Press the <ENTER> key. ');
    READLN(Ans);
  end;

begin                                            {Begin Main Program}
  Meth := 1;
  FunType := 1;
  P0 := 0;
  P1 := 1;
  P2 := 2;
  Stat := Working;
  while (Stat = Working) do
    begin
      MESSAGE(Meth, Delta, Epsilon);
      GETFUN(FunType);
      State := Computing;
      while (State = Computing) do
        begin
          GETPOINTS(P0, P1, P2, Meth);
          if Meth = 1 then
            begin
              STEFFEN(P0, Delta, Epsilon, Max, P3, Dp, Cond, K);
              RESULTS(P0, P3, Dp, Cond, K);
            end
          else
            begin
              MULLER(P0, P1, P2, Delta, Epsilon, Max, P3, Z, Satisfied);
              RESULTM(P0, P1, P2, P3, Z, K, Satisfied);
            end;
          WRITELN;
          WRITE('Do you want to see  all the approximations ?  <Y/N>  ');
          READLN(Ans);
          if (ANS = 'Y') or (ANS = 'y') then
            PRINTAPPROXS
          else
            WRITELN;
          case Meth of
            1: 
              WRITE('Want  to try  a  different  starting value ?  <Y/N>  ');
            2: 
              WRITE('Want  to  try  different  starting  values ?  <Y/N>  ');
          end;
          READLN(Ans);
          if (Ans <> 'Y') and (Ans <> 'y') then
            State := Done;
          if (Ans = 'Y') or (Ans = 'y') then
            CLRSCR;
        end;
      WRITELN;
      WRITE('Want to try a different function or method ?  <Y/N>  ');
      READLN(Ans);
      if (Ans <> 'Y') and (Ans <> 'y') then
        Stat := Done
    end;
end.                                               {End Main Program}

